/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.loader;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.util.TestingUtil;
import static org.testng.AssertJUnit.assertEquals;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

/**
 * A simple non-failing unit test to measure how many times each method on a cache loader is called.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
@Test(groups = "functional")
public class CacheLoaderMethodCallCounterTest extends AbstractCacheLoaderTestBase
{
   private CacheSPI cache;
   private DummyCountingCacheLoader dummyLoader;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      if (cache != null) tearDown();
      cache = (CacheSPI) new DefaultCacheFactory().createCache(false);
      cache.getConfiguration().setCacheLoaderConfig(getSingleCacheLoaderConfig("", DummyCountingCacheLoader.class.getName(), "", false, false, false));
      cache.start();
      dummyLoader = (DummyCountingCacheLoader) cache.getCacheLoaderManager().getCacheLoader();
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown()
   {
      TestingUtil.killCaches(cache);
   }


   public void testPut()
   {
      cache.put("/node", "key", "value");
      printReport("putKeyValue", dummyLoader);
   }

   public void testGet()
   {
      cache.get("/node", "key");
      printReport("getKey", dummyLoader);
   }

   public void testRemove()
   {
      cache.remove("/node", "key");
      printReport("removeKey", dummyLoader);
   }

   private void printReport(String test, DummyCountingCacheLoader d)
   {
      System.out.println("------------------------------");
      System.out.println(" Test name: " + test);
      System.out.println("  cache loader stats:");
      System.out.println("   put count: " + d.getPutCount());
      System.out.println("   get count: " + d.getGetCount());
      System.out.println("   exists count: " + d.getExistsCount());
      System.out.println("   remove count: " + d.getRemoveCount());
      System.out.println("   getChildrenNames count: " + d.getGetChildrenNamesCount());
      System.out.println("------------------------------");
   }

   public void testLoopedGets()
   {
      // put an object in cache
      cache.put("/test", "key", "value");

      // we should see this put in the cl
      assertEquals(1, dummyLoader.getPutCount());
      // the cloader interceptor does a get as well when doing the put ... ?
      assertEquals(1, dummyLoader.getGetCount());

      for (int i = 0; i < 2000; i++)
      {
         cache.getNode("/test");
      }

      assertEquals(1, dummyLoader.getPutCount());
      assertEquals(1, dummyLoader.getGetCount());
      assertEquals(0, dummyLoader.getRemoveCount());
      assertEquals(0, dummyLoader.getExistsCount());
   }
}