package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.DataContainer;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeNotExistsException;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.ReversibleCommand;
import org.jboss.cache.commands.Visitor;
import org.jboss.cache.commands.read.AbstractDataCommand;
import org.jboss.cache.notifications.Notifier;
import org.jboss.cache.transaction.GlobalTransaction;

/**
 * Implements functionality defined by {@link org.jboss.cache.Cache#move(org.jboss.cache.Fqn, org.jboss.cache.Fqn)}
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
// TODO: 2.2.0: Make sure this is properly intercepted, i.e., locked and loaded!!
public class MoveCommand extends AbstractDataCommand implements ReversibleCommand
{
   public static final int METHOD_ID = 36;
   private static final Log log = LogFactory.getLog(MoveCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /* dependencies */
   private Notifier notifier;

   /* params */
   private Fqn to;
   private GlobalTransaction globalTransaction;

   public MoveCommand()
   {
   }

   public void initialize(Notifier notifier, DataContainer dataContainer)
   {
      this.notifier = notifier;
      this.dataContainer = dataContainer;
   }

   public MoveCommand(Fqn from, Fqn to)
   {
      this.fqn = from;
      this.to = to;
   }

   public GlobalTransaction getGlobalTransaction()
   {
      return globalTransaction;
   }

   public void setGlobalTransaction(GlobalTransaction globalTransaction)
   {
      this.globalTransaction = globalTransaction;
   }

   /**
    * Moves a node, from <tt>fqn</tt> to <tt>to</tt>, and returns null.
    *
    * @param ctx invocation context
    * @return null
    */
   public Object perform(InvocationContext ctx)
   {
      move(fqn, to, false, ctx);
      return null;
   }

   public void rollback()
   {
      move(Fqn.fromRelativeElements(to, fqn.getLastElement()), fqn.getParent(), true, null);
   }


   private void adjustFqn(NodeSPI node, Fqn newBase)
   {
      Fqn newFqn = Fqn.fromRelativeElements(newBase, node.getFqn().getLastElement());
      node.setFqn(newFqn);
   }

   public Object acceptVisitor(InvocationContext ctx, Visitor visitor) throws Throwable
   {
      return visitor.visitMoveCommand(ctx, this);
   }

   private void move(Fqn toMoveFqn, Fqn newParentFqn, boolean skipNotifications, InvocationContext ctx)
   {
      // the actual move algorithm.
      NodeSPI newParent = dataContainer.peek(newParentFqn, false, false);
      if (newParent == null)
      {
         throw new NodeNotExistsException("New parent node " + newParentFqn + " does not exist when attempting to move node!!");
      }

      NodeSPI node = dataContainer.peek(toMoveFqn, false, false);

      if (node == null)
      {
         throw new NodeNotExistsException("Node " + toMoveFqn + " does not exist when attempting to move node!!");
      }

      if (trace) log.trace("Moving " + fqn + " to sit under " + to);

      NodeSPI oldParent = node.getParent();
      Object nodeName = toMoveFqn.getLastElement();

      // now that we have the parent and target nodes:
      // first correct the pointers at the pruning point
      oldParent.removeChildDirect(nodeName);
      newParent.addChild(nodeName, node);
      // parent pointer is calculated on the fly using Fqns.

      // notify
      if (!skipNotifications)
         notifier.notifyNodeMoved(toMoveFqn, Fqn.fromRelativeElements(newParentFqn, toMoveFqn.getLastElement()), true, ctx);

      // now adjust Fqns of node and all children.
      adjustFqn(node, newParent.getFqn());

      if (!skipNotifications)
         notifier.notifyNodeMoved(toMoveFqn, Fqn.fromRelativeElements(newParentFqn, toMoveFqn.getLastElement()), false, ctx);
   }

   public Fqn getTo()
   {
      return to;
   }

   public int getCommandId()
   {
      return METHOD_ID;
   }

   @Override
   public Object[] getParameters()
   {
      return new Object[]{fqn, to};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      fqn = (Fqn) args[0];
      to = (Fqn) args[1];
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;

      MoveCommand that = (MoveCommand) o;

      if (to != null ? !to.equals(that.to) : that.to != null) return false;

      return true;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + (to != null ? to.hashCode() : 0);
      return result;
   }

   @Override
   public String toString()
   {
      return "MoveCommand{" +
            "fqn=" + fqn +
            ", to=" + to +
            '}';
   }
}
