/*******************************************************************************
* Copyright (C) 2025 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*    Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Inspector-Executor
*              Sparse BLAS C example for the mkl_sparse_?_convert_dense
*              routine to convert from compressed sparse row (CSR) sparse matrix
*              format to a dense matrix representation.
*
********************************************************************************
*
*  Example program for using Intel(R) oneMKL Inspector-Executor Sparse BLAS
*  routines for conversion from sparse to dense matrix representation.
*
* The following Inspector-Executor Sparse BLAS routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_csr
*          mkl_sparse_destroy
*
*   Format conversion functions:
*          mkl_sparse_d_convert_dense
*
*
* Consider the matrix A below to be represented in compressed sparse row (CSR)
* format (see 'Sparse Matrix Storage Schemes' in the Intel(R) oneMKL C Reference
* Manual):
*
*       |  1  -2   0   0 |
*       |  3  -4   0   0 |
*   A = |  0   0   5  -6 |.
*       |  0   0   7  -8 |
*       |  9 -10   0   0 |
*       | 11 -12   0   0 |
*
*
* A CSR representation of the matrix with three arrays is given below:
*
*     csrNrows  = 6
*     csrNcols  = 4
*     csrNnz    = 12
*     csrIndex  = SPARSE_INDEX_BASE_ZERO
*     csrRowPtr = (0       2       4       6       8      10      12)
*     csrColIdx = (0   1   0   1   2   3   2   3   0   1   0   1)
*     csrValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
*
* This example presents:
*    * mkl_sparse_d_create_csr() usage to create a handle for a sparse matrix in
*      CSR format
*    * mkl_sparse_d_convert_dense() usage to convert a sparse matrix from CSR to
*      dense format
*    * Validation of the conversion result by comparing the converted dense
*      matrix with the reference one
*
* Note: The mkl_sparse_?_convert_dense() API supports conversion from an input
*       matrix in any of the following sparse formats:
*       - Compressed Sparse Row (CSR)
*       - Compressed Sparse Column (CSC)
*       - Block Compressed Sparse Row (BSR)
*       - Coordinate (COO)
*       The arguments for the call to mkl_sparse_?_convert_dense will be the same
*       for all those formats. The only difference will be in the call to the
*       initialization routine for the sparse matrix, which needs to match the
*       corresponding sparse format.
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "mkl.h"

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%4lld"
#else
#define INT_PRINT_FORMAT "%4d"
#endif

void print_int_value(const char *name, MKL_INT val) {
    printf("\t\t%s = " INT_PRINT_FORMAT "\n", name, val);
}

void print_int_array(const char *name, MKL_INT *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf(INT_PRINT_FORMAT ",", array[i]);
    }
    printf("\n");
}

void print_index(const char *name, sparse_index_base_t idxBase) {
    printf("\t\t%s = %s\n", name, idxBase == SPARSE_INDEX_BASE_ZERO ? "SPARSE_INDEX_BASE_ZERO" : "SPARSE_INDEX_BASE_ONE");
}

void print_flt_array(const char *name, double *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf("%4.0f,", array[i]);
    }
    printf("\n");
}

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the CSR format:
    //*******************************************************************************
#define M 6
#define N 4
#define NNZ 12

    sparse_matrix_t csrA = NULL;
    struct matrix_descr descrA;
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    sparse_index_base_t indexing = SPARSE_INDEX_BASE_ZERO;
    sparse_status_t status;
    int exit_status = 0;

    //*******************************************************************************
    //    Arrays for the CSR representation of the matrix A
    //*******************************************************************************

    MKL_INT csrRowPtr[M+1] = { 0, 2, 4, 6, 8, 10, 12 };
    MKL_INT csrColIdx[NNZ] = { 0, 1, 0, 1, 2, 3, 2, 3, 0, 1, 0, 1 };
    double csrValues[NNZ]  = { 1, -2, 3, -4, 5, -6, 7, -8, 9, -10, 11, -12 };

    //*******************************************************************************
    //    Dense representation of matrix A with row-major layout and leading
    //    dimension ldA = N, to be used for validation later
    //*******************************************************************************

    sparse_layout_t dense_layout = SPARSE_LAYOUT_ROW_MAJOR;
    MKL_INT ldA = N;
    double denseMatrixRef[M * N] = {  1,  -2,   0,   0,
                                      3,  -4,   0,   0,
                                      0,   0,   5,  -6,
                                      0,   0,   7,  -8,
                                      9, -10,   0,   0,
                                     11, -12,   0,   0 };

    //*******************************************************************************
    //     Dense representation of matrix A to be filled by the conversion routine,
    //     with row-major layout and leading dimension ldA = N
    //*******************************************************************************

    double denseMatrix[M * N];

    printf( "\nExample program for conversion from a sparse matrix in CSR format to\n" );
    printf( " a dense matrix using IE Sparse BLAS APIs\n" );
    printf( "-------------------------------------------------------------------------\n" );

    //******************************************************************************
    //    Create sparse matrix handle: csrA
    //******************************************************************************
    status = mkl_sparse_d_create_csr(&csrA, indexing, M, N, csrRowPtr,
                                     csrRowPtr+1, csrColIdx, csrValues);

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_csr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("\t[Input] Matrix arrays in CSR format:\n");
    print_int_value("nrows", M);
    print_int_value("ncols", N);
    print_int_value("nnz  ", NNZ);
    print_index(    "index", indexing);
    print_int_array("csrRowIdx", csrRowPtr, M+1);
    print_int_array("csrColIdx", csrColIdx, NNZ);
    print_flt_array("csrValues", csrValues, NNZ);
    printf("\n");

    //******************************************************************************
    //    Convert from CSR to DENSE format
    //******************************************************************************
    status = mkl_sparse_d_convert_dense(csrA, descrA, dense_layout, ldA, denseMatrix);

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_convert_dense: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("\t[Output] Matrix array in dense format (converted from CSR):\n");
    print_int_value("nrows", M);
    print_int_value("ncols", N);
    printf("\t\tdenseMatrix:\n");
    for (MKL_INT i = 0; i < M; ++i) {
        printf("\t\t");
        for (MKL_INT j = 0; j < N; ++j) {
            printf(" %6.0lf ", denseMatrix[i * N + j]);
        }
        printf("\n");
    }
    printf("\n");

    //******************************************************************************
    //    Validation of the conversion result by comparing with the reference
    //    Note that the conversion should be exact and that we do not compare
    //    the result with some tolerance
    //******************************************************************************
    int conversion_passed = 1;
    for (MKL_INT i = 0; i < M*N; ++i) {
        if (denseMatrix[i] != denseMatrixRef[i]) {
            conversion_passed = 0;
            break;
        }
    }

    if (conversion_passed) {
        printf("\t The conversion from CSR format to dense passed\n");
    }
    else
    {
        printf("\t The conversion from CSR format to dense failed\n");
        exit_status = 1;
        goto exit;
    }
    printf("\n");

exit:
    // Release matrix handle
    if (csrA) mkl_sparse_destroy(csrA);

    return exit_status;
}
