﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/guardduty/GuardDutyRequest.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>
#include <aws/guardduty/model/DestinationProperties.h>
#include <aws/guardduty/model/DestinationType.h>

#include <utility>

namespace Aws {
namespace GuardDuty {
namespace Model {

/**
 */
class CreatePublishingDestinationRequest : public GuardDutyRequest {
 public:
  AWS_GUARDDUTY_API CreatePublishingDestinationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePublishingDestination"; }

  AWS_GUARDDUTY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the GuardDuty detector associated with the publishing
   * destination.</p> <p>To find the <code>detectorId</code> in the current Region,
   * see the Settings page in the GuardDuty console, or run the <a
   * href="https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html">ListDetectors</a>
   * API.</p>
   */
  inline const Aws::String& GetDetectorId() const { return m_detectorId; }
  inline bool DetectorIdHasBeenSet() const { return m_detectorIdHasBeenSet; }
  template <typename DetectorIdT = Aws::String>
  void SetDetectorId(DetectorIdT&& value) {
    m_detectorIdHasBeenSet = true;
    m_detectorId = std::forward<DetectorIdT>(value);
  }
  template <typename DetectorIdT = Aws::String>
  CreatePublishingDestinationRequest& WithDetectorId(DetectorIdT&& value) {
    SetDetectorId(std::forward<DetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of resource for the publishing destination. Currently only Amazon S3
   * buckets are supported.</p>
   */
  inline DestinationType GetDestinationType() const { return m_destinationType; }
  inline bool DestinationTypeHasBeenSet() const { return m_destinationTypeHasBeenSet; }
  inline void SetDestinationType(DestinationType value) {
    m_destinationTypeHasBeenSet = true;
    m_destinationType = value;
  }
  inline CreatePublishingDestinationRequest& WithDestinationType(DestinationType value) {
    SetDestinationType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The properties of the publishing destination, including the ARNs for the
   * destination and the KMS key used for encryption.</p>
   */
  inline const DestinationProperties& GetDestinationProperties() const { return m_destinationProperties; }
  inline bool DestinationPropertiesHasBeenSet() const { return m_destinationPropertiesHasBeenSet; }
  template <typename DestinationPropertiesT = DestinationProperties>
  void SetDestinationProperties(DestinationPropertiesT&& value) {
    m_destinationPropertiesHasBeenSet = true;
    m_destinationProperties = std::forward<DestinationPropertiesT>(value);
  }
  template <typename DestinationPropertiesT = DestinationProperties>
  CreatePublishingDestinationRequest& WithDestinationProperties(DestinationPropertiesT&& value) {
    SetDestinationProperties(std::forward<DestinationPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The idempotency token for the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreatePublishingDestinationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_detectorId;
  bool m_detectorIdHasBeenSet = false;

  DestinationType m_destinationType{DestinationType::NOT_SET};
  bool m_destinationTypeHasBeenSet = false;

  DestinationProperties m_destinationProperties;
  bool m_destinationPropertiesHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
