﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/inspector2/Inspector2_EXPORTS.h>
#include <aws/inspector2/model/AtigData.h>
#include <aws/inspector2/model/CisaData.h>
#include <aws/inspector2/model/Cvss2.h>
#include <aws/inspector2/model/Cvss3.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector2/model/Epss.h>
#include <aws/inspector2/model/ExploitObserved.h>
#include <aws/inspector2/model/VulnerabilitySource.h>
#include <aws/core/utils/DateTime.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace Inspector2
{
namespace Model
{

  /**
   * <p>Contains details about a specific vulnerability Amazon Inspector can
   * detect.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/inspector2-2020-06-08/Vulnerability">AWS
   * API Reference</a></p>
   */
  class Vulnerability
  {
  public:
    AWS_INSPECTOR2_API Vulnerability() = default;
    AWS_INSPECTOR2_API Vulnerability(Aws::Utils::Json::JsonView jsonValue);
    AWS_INSPECTOR2_API Vulnerability& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_INSPECTOR2_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>An object that contains information about the Amazon Web Services Threat
     * Intel Group (ATIG) details for the vulnerability.</p>
     */
    inline const AtigData& GetAtigData() const { return m_atigData; }
    inline bool AtigDataHasBeenSet() const { return m_atigDataHasBeenSet; }
    template<typename AtigDataT = AtigData>
    void SetAtigData(AtigDataT&& value) { m_atigDataHasBeenSet = true; m_atigData = std::forward<AtigDataT>(value); }
    template<typename AtigDataT = AtigData>
    Vulnerability& WithAtigData(AtigDataT&& value) { SetAtigData(std::forward<AtigDataT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>An object that contains the Cybersecurity and Infrastructure Security Agency
     * (CISA) details for the vulnerability.</p>
     */
    inline const CisaData& GetCisaData() const { return m_cisaData; }
    inline bool CisaDataHasBeenSet() const { return m_cisaDataHasBeenSet; }
    template<typename CisaDataT = CisaData>
    void SetCisaData(CisaDataT&& value) { m_cisaDataHasBeenSet = true; m_cisaData = std::forward<CisaDataT>(value); }
    template<typename CisaDataT = CisaData>
    Vulnerability& WithCisaData(CisaDataT&& value) { SetCisaData(std::forward<CisaDataT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>An object that contains the Common Vulnerability Scoring System (CVSS)
     * Version 2 details for the vulnerability.</p>
     */
    inline const Cvss2& GetCvss2() const { return m_cvss2; }
    inline bool Cvss2HasBeenSet() const { return m_cvss2HasBeenSet; }
    template<typename Cvss2T = Cvss2>
    void SetCvss2(Cvss2T&& value) { m_cvss2HasBeenSet = true; m_cvss2 = std::forward<Cvss2T>(value); }
    template<typename Cvss2T = Cvss2>
    Vulnerability& WithCvss2(Cvss2T&& value) { SetCvss2(std::forward<Cvss2T>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>An object that contains the Common Vulnerability Scoring System (CVSS)
     * Version 3 details for the vulnerability.</p>
     */
    inline const Cvss3& GetCvss3() const { return m_cvss3; }
    inline bool Cvss3HasBeenSet() const { return m_cvss3HasBeenSet; }
    template<typename Cvss3T = Cvss3>
    void SetCvss3(Cvss3T&& value) { m_cvss3HasBeenSet = true; m_cvss3 = std::forward<Cvss3T>(value); }
    template<typename Cvss3T = Cvss3>
    Vulnerability& WithCvss3(Cvss3T&& value) { SetCvss3(std::forward<Cvss3T>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The Common Weakness Enumeration (CWE) associated with the vulnerability.</p>
     */
    inline const Aws::Vector<Aws::String>& GetCwes() const { return m_cwes; }
    inline bool CwesHasBeenSet() const { return m_cwesHasBeenSet; }
    template<typename CwesT = Aws::Vector<Aws::String>>
    void SetCwes(CwesT&& value) { m_cwesHasBeenSet = true; m_cwes = std::forward<CwesT>(value); }
    template<typename CwesT = Aws::Vector<Aws::String>>
    Vulnerability& WithCwes(CwesT&& value) { SetCwes(std::forward<CwesT>(value)); return *this;}
    template<typename CwesT = Aws::String>
    Vulnerability& AddCwes(CwesT&& value) { m_cwesHasBeenSet = true; m_cwes.emplace_back(std::forward<CwesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A description of the vulnerability.</p>
     */
    inline const Aws::String& GetDescription() const { return m_description; }
    inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
    template<typename DescriptionT = Aws::String>
    void SetDescription(DescriptionT&& value) { m_descriptionHasBeenSet = true; m_description = std::forward<DescriptionT>(value); }
    template<typename DescriptionT = Aws::String>
    Vulnerability& WithDescription(DescriptionT&& value) { SetDescription(std::forward<DescriptionT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Platforms that the vulnerability can be detected on.</p>
     */
    inline const Aws::Vector<Aws::String>& GetDetectionPlatforms() const { return m_detectionPlatforms; }
    inline bool DetectionPlatformsHasBeenSet() const { return m_detectionPlatformsHasBeenSet; }
    template<typename DetectionPlatformsT = Aws::Vector<Aws::String>>
    void SetDetectionPlatforms(DetectionPlatformsT&& value) { m_detectionPlatformsHasBeenSet = true; m_detectionPlatforms = std::forward<DetectionPlatformsT>(value); }
    template<typename DetectionPlatformsT = Aws::Vector<Aws::String>>
    Vulnerability& WithDetectionPlatforms(DetectionPlatformsT&& value) { SetDetectionPlatforms(std::forward<DetectionPlatformsT>(value)); return *this;}
    template<typename DetectionPlatformsT = Aws::String>
    Vulnerability& AddDetectionPlatforms(DetectionPlatformsT&& value) { m_detectionPlatformsHasBeenSet = true; m_detectionPlatforms.emplace_back(std::forward<DetectionPlatformsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>An object that contains the Exploit Prediction Scoring System (EPSS) score
     * for a vulnerability.</p>
     */
    inline const Epss& GetEpss() const { return m_epss; }
    inline bool EpssHasBeenSet() const { return m_epssHasBeenSet; }
    template<typename EpssT = Epss>
    void SetEpss(EpssT&& value) { m_epssHasBeenSet = true; m_epss = std::forward<EpssT>(value); }
    template<typename EpssT = Epss>
    Vulnerability& WithEpss(EpssT&& value) { SetEpss(std::forward<EpssT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>An object that contains details on when the exploit was observed.</p>
     */
    inline const ExploitObserved& GetExploitObserved() const { return m_exploitObserved; }
    inline bool ExploitObservedHasBeenSet() const { return m_exploitObservedHasBeenSet; }
    template<typename ExploitObservedT = ExploitObserved>
    void SetExploitObserved(ExploitObservedT&& value) { m_exploitObservedHasBeenSet = true; m_exploitObserved = std::forward<ExploitObservedT>(value); }
    template<typename ExploitObservedT = ExploitObserved>
    Vulnerability& WithExploitObserved(ExploitObservedT&& value) { SetExploitObserved(std::forward<ExploitObservedT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The ID for the specific vulnerability.</p>
     */
    inline const Aws::String& GetId() const { return m_id; }
    inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
    template<typename IdT = Aws::String>
    void SetId(IdT&& value) { m_idHasBeenSet = true; m_id = std::forward<IdT>(value); }
    template<typename IdT = Aws::String>
    Vulnerability& WithId(IdT&& value) { SetId(std::forward<IdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Links to various resources with more information on this vulnerability. </p>
     */
    inline const Aws::Vector<Aws::String>& GetReferenceUrls() const { return m_referenceUrls; }
    inline bool ReferenceUrlsHasBeenSet() const { return m_referenceUrlsHasBeenSet; }
    template<typename ReferenceUrlsT = Aws::Vector<Aws::String>>
    void SetReferenceUrls(ReferenceUrlsT&& value) { m_referenceUrlsHasBeenSet = true; m_referenceUrls = std::forward<ReferenceUrlsT>(value); }
    template<typename ReferenceUrlsT = Aws::Vector<Aws::String>>
    Vulnerability& WithReferenceUrls(ReferenceUrlsT&& value) { SetReferenceUrls(std::forward<ReferenceUrlsT>(value)); return *this;}
    template<typename ReferenceUrlsT = Aws::String>
    Vulnerability& AddReferenceUrls(ReferenceUrlsT&& value) { m_referenceUrlsHasBeenSet = true; m_referenceUrls.emplace_back(std::forward<ReferenceUrlsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A list of related vulnerabilities.</p>
     */
    inline const Aws::Vector<Aws::String>& GetRelatedVulnerabilities() const { return m_relatedVulnerabilities; }
    inline bool RelatedVulnerabilitiesHasBeenSet() const { return m_relatedVulnerabilitiesHasBeenSet; }
    template<typename RelatedVulnerabilitiesT = Aws::Vector<Aws::String>>
    void SetRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { m_relatedVulnerabilitiesHasBeenSet = true; m_relatedVulnerabilities = std::forward<RelatedVulnerabilitiesT>(value); }
    template<typename RelatedVulnerabilitiesT = Aws::Vector<Aws::String>>
    Vulnerability& WithRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { SetRelatedVulnerabilities(std::forward<RelatedVulnerabilitiesT>(value)); return *this;}
    template<typename RelatedVulnerabilitiesT = Aws::String>
    Vulnerability& AddRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { m_relatedVulnerabilitiesHasBeenSet = true; m_relatedVulnerabilities.emplace_back(std::forward<RelatedVulnerabilitiesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source of the vulnerability information. Possible results are
     * <code>RHEL</code>, <code>AMAZON_CVE</code>, <code>DEBIAN</code> or
     * <code>NVD</code>.</p>
     */
    inline VulnerabilitySource GetSource() const { return m_source; }
    inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
    inline void SetSource(VulnerabilitySource value) { m_sourceHasBeenSet = true; m_source = value; }
    inline Vulnerability& WithSource(VulnerabilitySource value) { SetSource(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>A link to the official source material for this vulnerability.</p>
     */
    inline const Aws::String& GetSourceUrl() const { return m_sourceUrl; }
    inline bool SourceUrlHasBeenSet() const { return m_sourceUrlHasBeenSet; }
    template<typename SourceUrlT = Aws::String>
    void SetSourceUrl(SourceUrlT&& value) { m_sourceUrlHasBeenSet = true; m_sourceUrl = std::forward<SourceUrlT>(value); }
    template<typename SourceUrlT = Aws::String>
    Vulnerability& WithSourceUrl(SourceUrlT&& value) { SetSourceUrl(std::forward<SourceUrlT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The date and time when the vendor created this vulnerability.</p>
     */
    inline const Aws::Utils::DateTime& GetVendorCreatedAt() const { return m_vendorCreatedAt; }
    inline bool VendorCreatedAtHasBeenSet() const { return m_vendorCreatedAtHasBeenSet; }
    template<typename VendorCreatedAtT = Aws::Utils::DateTime>
    void SetVendorCreatedAt(VendorCreatedAtT&& value) { m_vendorCreatedAtHasBeenSet = true; m_vendorCreatedAt = std::forward<VendorCreatedAtT>(value); }
    template<typename VendorCreatedAtT = Aws::Utils::DateTime>
    Vulnerability& WithVendorCreatedAt(VendorCreatedAtT&& value) { SetVendorCreatedAt(std::forward<VendorCreatedAtT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The severity assigned by the vendor.</p>
     */
    inline const Aws::String& GetVendorSeverity() const { return m_vendorSeverity; }
    inline bool VendorSeverityHasBeenSet() const { return m_vendorSeverityHasBeenSet; }
    template<typename VendorSeverityT = Aws::String>
    void SetVendorSeverity(VendorSeverityT&& value) { m_vendorSeverityHasBeenSet = true; m_vendorSeverity = std::forward<VendorSeverityT>(value); }
    template<typename VendorSeverityT = Aws::String>
    Vulnerability& WithVendorSeverity(VendorSeverityT&& value) { SetVendorSeverity(std::forward<VendorSeverityT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The date and time when the vendor last updated this vulnerability.</p>
     */
    inline const Aws::Utils::DateTime& GetVendorUpdatedAt() const { return m_vendorUpdatedAt; }
    inline bool VendorUpdatedAtHasBeenSet() const { return m_vendorUpdatedAtHasBeenSet; }
    template<typename VendorUpdatedAtT = Aws::Utils::DateTime>
    void SetVendorUpdatedAt(VendorUpdatedAtT&& value) { m_vendorUpdatedAtHasBeenSet = true; m_vendorUpdatedAt = std::forward<VendorUpdatedAtT>(value); }
    template<typename VendorUpdatedAtT = Aws::Utils::DateTime>
    Vulnerability& WithVendorUpdatedAt(VendorUpdatedAtT&& value) { SetVendorUpdatedAt(std::forward<VendorUpdatedAtT>(value)); return *this;}
    ///@}
  private:

    AtigData m_atigData;
    bool m_atigDataHasBeenSet = false;

    CisaData m_cisaData;
    bool m_cisaDataHasBeenSet = false;

    Cvss2 m_cvss2;
    bool m_cvss2HasBeenSet = false;

    Cvss3 m_cvss3;
    bool m_cvss3HasBeenSet = false;

    Aws::Vector<Aws::String> m_cwes;
    bool m_cwesHasBeenSet = false;

    Aws::String m_description;
    bool m_descriptionHasBeenSet = false;

    Aws::Vector<Aws::String> m_detectionPlatforms;
    bool m_detectionPlatformsHasBeenSet = false;

    Epss m_epss;
    bool m_epssHasBeenSet = false;

    ExploitObserved m_exploitObserved;
    bool m_exploitObservedHasBeenSet = false;

    Aws::String m_id;
    bool m_idHasBeenSet = false;

    Aws::Vector<Aws::String> m_referenceUrls;
    bool m_referenceUrlsHasBeenSet = false;

    Aws::Vector<Aws::String> m_relatedVulnerabilities;
    bool m_relatedVulnerabilitiesHasBeenSet = false;

    VulnerabilitySource m_source{VulnerabilitySource::NOT_SET};
    bool m_sourceHasBeenSet = false;

    Aws::String m_sourceUrl;
    bool m_sourceUrlHasBeenSet = false;

    Aws::Utils::DateTime m_vendorCreatedAt{};
    bool m_vendorCreatedAtHasBeenSet = false;

    Aws::String m_vendorSeverity;
    bool m_vendorSeverityHasBeenSet = false;

    Aws::Utils::DateTime m_vendorUpdatedAt{};
    bool m_vendorUpdatedAtHasBeenSet = false;
  };

} // namespace Model
} // namespace Inspector2
} // namespace Aws
