#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

"""
This module provides access to Intel Advisor results, collected while running the Intel Advisor analysis.

Getting Started
+++++++++++++++

Before you begin, configure the environment:

1. Add path to the pythonapi directory into PYTHONPATH environment variable, or use advixe-vars.* scripts to set up product environment variables automatically. The scripts reside in the Advisor installation directory root

2. Create a new project or open an existing project. These actions are similar to those in the Advisor GUI.
	2.a. You can create a project using the :func:`create_project`.
		>>> project = advisor.create_project(sys.argv[1])
	2.b. If project already exists, you can open a project using the the :func:`open_project`.
		>>> project = advisor.open_project(sys.argv[1])

3. Run the required collection using the :meth:`Project.collect` method. The following collections are available:

    * :attr:`advisor.SURVEY`
    * :attr:`advisor.TRIPCOUNTS`
    * :attr:`advisor.MAP`
    * :attr:`advisor.DEPENDENCIES`
    * :attr:`advisor.ROOFLINE`

       >>> res = project.collect(advisor.SURVEY, sys.argv[1])
    This action is similar to those in the Advisor GUI and the Advisor Command Line.
    After collection you do not need to reopen the project.

4. Load the result using the :meth:`Project.load` method. The following reports are available:

 * :attr:`advisor.SURVEY`
 * :attr:`advisor.MAP`
 * :attr:`advisor.DEPENDENCIES`
 * :attr:`advisor.VECTORIZATION_DATA`
 * :attr:`advisor.PERFORMANCE_PROJECTION`
 * :attr:`advisor.ALL`

    >>> survey = project.load(advisor.SURVEY)

If you use the :attr:`advisor.VECTORIZATION_DATA` report type, the Python API will load data of `advisor.SURVEY`, `advisor.MAP`, `advisor.DEPENDENCIES` if available. In addition, it will merge map and dependencies data into the survey bottom-up rows. To check traits and recommendatioins, use :attr:`advisor.VECTORIZATION_DATA` too.
If you use the :attr:`advisor.ALL` report type, the Python API will load all available data. As an `advisor.VECTORIZATION_DATA`, it will merge map and dependencies data into the survey bottom-up rows. To check traits and recommendatioins, you can also use :attr:`advisor.ALL`.

Accessing Data
^^^^^^^^^^^^^^

To access data, use row iterators:

* :attr:`DataModel.bottomup` if `advisor.SURVEY` or `advisor.VECTORIZATION_DATA` or `advisor.ALL` is loaded
* :attr:`DataModel.topdown` if `advisor.SURVEY` or `advisor.VECTORIZATION_DATA` or `advisor.ALL` is loaded
* :attr:`DataModel.map` if `advisor.MAP` is loaded
* :attr:`DataModel.dependencies` if `advisor.DEPENDENCIES` is loaded
* :attr:`DataModel.perf_projection` if `advisor.PERFORMANCE_PROJECTION` or `advisor.ALL` is loaded

Data Keys and Values
^^^^^^^^^^^^^^^^^^^^

You can access each data row (survey, map, or dependencies) as an associative array.

Print data row as string:
    >>> for row in survey.bottomup:
    >>>     print(row)

Get all row keys and values:
    >>> for key in row:
    ...     print(key, row[key])

Survey Data
^^^^^^^^^^^

Each :class:`Hotspot` row has :attr:`Hotspot.parent` attribute and :attr:`Hotspot.children`,
:attr:`Hotspot.assembly`, :attr:`Hotspot.sync` iterators and :attr:`Hotspot.basic_blocks` iterators.

Bottom-up is flat, parent and children are empty for all rows.

To print the top-down tree, do the following:
    >>> for row in survey.topdown:
    ...     stack = [(row, 0)]
    ...     while stack:
    ...         v, level = stack.pop()
    ...         for c in v.get_children():
    ...             stack.append((c, level + 1))
    ...             print('-' * level + v['function_call_sites_and_loops'])

:attr:`Hotspot.assembly` iterator contains assembly for given row with additional info on
arguments and instruction class.

:attr:`Hotspot.basic_blocks` iterator contains basic blocks for given row with additional info.
:attr:`Hotspot.basic_blocks_lite` iterator contains basic blocks for given row without additional info.

Get the required instruction mix information using the :meth:`Hotspot.get_instruction_mix` method. The following types are available:

* :attr:`advisor.MixType.STATIC_SELF` for exclusive (self) statically calculated instruction mix, the same as :attr:`Hotspot.static_self_mix`
* :attr:`advisor.MixType.DYNAMIC_SELF` for exclusive (self) dynamically calculated instruction mix, the same as :attr:`Hotspot.dynamic_self_mix`
* :attr:`advisor.MixType.DYNAMIC_SELF_AGGREGATED` for exclusive (self) dynamically calculated instruction mix including top-down connected metrics, the same as :attr:`Hotspot.dynamic_self_mix`
* :attr:`advisor.MixType.DYNAMIC_TOTAL_AGGREGATED` for inclusive (total) dynamically calculated instruction mix including top-down connected metrics, the same as :attr:`Hotspot.dynamic_total_mix`

:attr:`Hotspot.sync` iterator contains rows from another dataset, corresponding to
this row. For example, if row is from bottom-up then row.sync is an iterator over all
entries of this row in top-down.

Joined Survey and Refinement Data
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

If you load `advisor.ALL` report survey, map, and dependencies data will be available thru the :attr:`DataModel.bottomup` iterator.

Each :class:`Hotspot` row has keys for map and dependencies analysis values:
    >>> for row in data.bottomup:
    ...     print(row['loop_carries_dependencies'])

You can access map and dependencies sites using the :attr:`Hotspot.map` and :attr:`Hotspot.dependencies` iterators:
    >>> for row in data.bottomup:
    ...     for site in row.map:
    ...         for problem in site.problems:
    ...             print(problem)

Map and Dependencies Data
^^^^^^^^^^^^^^^^^^^^^^^^^

Each Map and Dependencies :class:`Site` row has the :attr:`Site.problems` iterator.

Each :class:`Problem` row has the :attr:`Problem.observations` iterator (empty for MAP result).


    >>> for site in data.dependencies:
    ...     print(site)
    ...     for problem in site.problems:
    ...         print(problem)
    ...         for observation in problem.observations:
    ...             print(observation)

Performance Projection Data
^^^^^^^^^^^^^^^^^^^^^^^^^

Each :class:`PerfProjectionRow` row has an :attr:`PerfProjectionRow.parent` attribute and an :attr:`PerfProjectionRow.children` iterator.

    >>> for top_row in data.perf_projection:
    ...     stack = [(top_row, 0)]
    ...     while stack:
    ...         row, level = stack.pop()
    ...         for c in row.children:
    ...             stack.append((c, level + 1))
    ...         print(('-' * level, row['source_location'], row['is_offloaded'], row['speedup'])
"""

from __future__ import absolute_import

import os
import sys

from .compat import deprecated

python_api_dir = os.path.dirname(os.path.realpath(__file__))  # current dir
lib_dir = os.path.join(python_api_dir, '..', '..')  # product root dir
platform_bit_width = 64 if sys.maxsize > 2**32 else 32
if os.name == 'posix':
    lib_dir = os.path.join(lib_dir, 'lib'+str(platform_bit_width))
else:
    lib_dir = os.path.join(lib_dir, 'bin'+str(platform_bit_width))
sys.path.append(lib_dir)  # adding path to advixe_discpythonapi[_major_minor] to PYTHONPATH

# Workaround for conflict between Advisor's and system libstdc++ libraries.
# That conflict prevents importing advisor and matplotlib in the same script.
try:
    if os.name == 'posix':
        import re
        import subprocess
        pattern = 'libstdc\+\+\.so\.6\.([0-9]+)\.([0-9]+)$'

        product_version = None
        product_lib = ''
        for lib in os.listdir(lib_dir):
            lib = lib.strip()
            match = re.match(pattern, lib)
            if match:
                product_version = (match.group(1), match.group(2))
                product_lib = lib
                break

        system_version = None
        # find system libstdc++ library full path and resolve symlink
        with open(os.devnull, 'w') as FNULL:
            ldconfig = 'ldconfig'
            if subprocess.run(['which', 'ldconfig'], stdout=FNULL, stderr=FNULL).returncode != 0:
                ldconfig = '/sbin/ldconfig'

        with open(os.devnull, 'w') as FNULL:
            result = subprocess.run([ldconfig, '-p'], stdout=subprocess.PIPE, stderr=FNULL)
            out = result.stdout

        system_lib = None
        if result.returncode == 0:
            lines = out.decode().splitlines()
            for line in lines:
                if 'libstdc++.so.6' in line:
                    parts = line.split()
                    if parts:
                        system_lib = parts[-1].strip()
                        break

        if system_lib:
            match = re.match(pattern, system_lib)
            if match:
                system_version = (match.group(1), match.group(2))

            if product_version and system_version:
                if product_version[0] == system_version[0]:
                    import ctypes
                    if product_version[1] >= system_version[1]:
                        ctypes.CDLL(os.path.join(lib_dir, product_lib))
                    else:
                        ctypes.CDLL(system_lib)
                    del ctypes
        del re
        del subprocess
except:
    pass

#if sys.version_info.major == 2:
#    import advixe_discpython2api1 as discpythonapi
import platform
if platform.system() == 'Windows' and 'exe' in sys.executable.lower():
    # We are running under external Python on Windows. Need library
    # exactly matching Python's major and minor versions
    MAJOR, MINOR, _, _, _ = sys.version_info
    import importlib
    discpythonapi = importlib.import_module("advixe_discpythonapi_{}_{}".format(MAJOR, MINOR))
    del importlib
else:
    import advixe_discpythonapi1 as discpythonapi

del sys
del platform
del lib_dir

advixe_cl_path = os.path.join(python_api_dir, '..', '..', 'bin64', 'advixe-cl')

del python_api_dir
del os


doc = """
Opens the Intel Advisor project

Args:
    project_path (str): absolute or relative path to project directory
    mpi_rank (int): MPI rank to open. If -1 (default) and there are several MPI ranks in the project then rank with highest number is selected.
Returns:
    :class:`Project` instance
"""
open_project = discpythonapi.open_project
open_project.__doc__ = doc
doc = """
Creates the Intel Advisor project

Args:
    project_path (str): absolute or relative path to project directory
Returns:
    :class:`Project` instance
"""
create_project = discpythonapi.create_project
create_project.__doc__ = doc

DataModel = discpythonapi.DataModel
Project = discpythonapi.Project
Observation = discpythonapi.Observation
Problem = discpythonapi.Problem
Site = discpythonapi.Site
ProgramMetrics = discpythonapi.ProgramMetrics
RowMatchInfo = discpythonapi.RowMatchInfo
RowMatchInfos = discpythonapi.RowMatchInfos

Project.LOCK = discpythonapi.OpenMode.LOCK
Project.NONLOCK = discpythonapi.OpenMode.NONLOCK

class SelectionTarget(object):
    CollectingFeature = discpythonapi.CollectingFeature
    TRIPCOUNTS = discpythonapi.CollectingFeature.TRIPCOUNTS
    FLOPS = discpythonapi.CollectingFeature.FLOPS
    CACHESIM = discpythonapi.CollectingFeature.CACHESIM
    DATA_TRANSFER = discpythonapi.CollectingFeature.DATA_TRANSFER
    MAP = discpythonapi.CollectingFeature.MAP
    DEPENDENCY = discpythonapi.CollectingFeature.DEPENDENCY
    PROJECTION = discpythonapi.CollectingFeature.PROJECTION
    ALL = discpythonapi.CollectingFeature.ALL


SURVEY = discpythonapi.ModelType.SURVEY
TRIPCOUNTS = discpythonapi.ModelType.TRIPCOUNTS
DEPENDENCIES = discpythonapi.ModelType.DEPENDENCIES
MAP = discpythonapi.ModelType.MAP
ROOFLINE = discpythonapi.ModelType.ROOFLINE
RECOMMENDATIONS = discpythonapi.ModelType.RECOMMENDATIONS
VECTORIZATION_DATA = discpythonapi.ModelType.VECTORIZATION_DATA
PERFORMANCE_PROJECTION = discpythonapi.ModelType.PERFORMANCE_PROJECTION
ALL = discpythonapi.ModelType.ALL


class MixTarget(object):
    CATEGORY = discpythonapi.MixTarget.CATEGORY
    TYPE = discpythonapi.MixTarget.TYPE
    ISA = discpythonapi.MixTarget.ISA
    OPERAND = discpythonapi.MixTarget.OPERAND
    DATATYPE = discpythonapi.MixTarget.DATATYPE

from . import mix

class MixType(object):
    STATIC_SELF = "static_self"
    STATIC_TOTAL = "static_total"
    DYNAMIC_SELF = "dynamic_self"
    DYNAMIC_TOTAL = "dynamic_total"

InstructionType = discpythonapi.InstructionType


class InternalMixType(object):
    STATIC_SELF = discpythonapi.InstructionMixType.STATIC_SELF
    DYNAMIC_SELF = discpythonapi.InstructionMixType.DYNAMIC_SELF
    DYNAMIC_SELF_AGGREGATED = discpythonapi.InstructionMixType.DYNAMIC_SELF_AGGREGATED
    DYNAMIC_TOTAL_AGGREGATED = discpythonapi.InstructionMixType.DYNAMIC_TOTAL_AGGREGATED

def get_instruction_mix(slf, mix_type):
    """
    Get iterator for instruction mix of given type. Usually it is just one line,
    containing instructions breakdown for loop or function.

    Args:
        mix_type: one of the constants from :class:`MixType`
    Returns:
        :class:`InstructionMixIterator` instance
    """
    res = None
    if mix_type == MixType.STATIC_TOTAL:
        res = mix.InstructionMixIterator(mix.get_static_total_mix(slf, MixType.STATIC_SELF), 'static_')
    if mix_type == MixType.DYNAMIC_SELF:
        topdown_dynamic_self = slf._get_instruction_mix(InternalMixType.DYNAMIC_SELF_AGGREGATED)
        if not next(topdown_dynamic_self, None):
            res = mix.InstructionMixIterator(slf._get_instruction_mix(InternalMixType.DYNAMIC_SELF), 'dynamic_')
        else:
            res = mix.InstructionMixIterator(slf._get_instruction_mix(InternalMixType.DYNAMIC_SELF_AGGREGATED), 'self_')
    if mix_type == MixType.STATIC_SELF:
        res = mix.InstructionMixIterator(slf._get_instruction_mix(InternalMixType.STATIC_SELF), 'static_')
    if mix_type == MixType.DYNAMIC_TOTAL:
        res = mix.InstructionMixIterator(slf._get_instruction_mix(InternalMixType.DYNAMIC_TOTAL_AGGREGATED), 'total_')
    return res

class RoofsStrategy(object):
    MULTI_THREAD = discpythonapi.RoofsStrategy.MULTI_THREAD
    SINGLE_THREAD = discpythonapi.RoofsStrategy.SINGLE_THREAD

class GpuDataType(object):
    BASIC = discpythonapi.DataType.GPU
    ROOFLINE = discpythonapi.DataType.GPU_ROOFLINE
    INSTRUCTION_MIX = discpythonapi.DataType.GPU_INSTRUCTION_MIX
    ROOFLINE_INSTRUCTION_MIX = discpythonapi.DataType.GPU_ROOFLINE_INSTRUCTION_MIX

class AdditionalDataType(object):
    INSTRUCTION_MIX = discpythonapi.DataType.INSTRUCTION_MIX
    MEMORY_TRAFFIC = discpythonapi.DataType.MEMORY_TRAFFIC
    MEMORY_OBJECTS = discpythonapi.DataType.MEMORY_OBJECTS
    CPU_KERNEL_INFO = discpythonapi.DataType.CPU_KERNEL_INFO

class ResultType(object):
    Survey = discpythonapi.ResultType.Survey
    Suitability = discpythonapi.ResultType.Suitability
    Correctness = discpythonapi.ResultType.Correctness
    MAP = discpythonapi.ResultType.MAP
    TripCounts = discpythonapi.ResultType.TripCounts
    DCFG = discpythonapi.ResultType.DCFG
    PerfProjection = discpythonapi.ResultType.PerfProjection
    NumResults = discpythonapi.ResultType.NumResults

class RowType(object):
    LOOP = discpythonapi.RowType.LOOP
    INNER_LOOP = discpythonapi.RowType.INNER_LOOP
    PEEL_LOOP = discpythonapi.RowType.PEEL_LOOP
    REMAINDER_LOOP = discpythonapi.RowType.REMAINDER_LOOP
    BODY_LOOP = discpythonapi.RowType.BODY_LOOP
    VECTORIZED = discpythonapi.RowType.VECTORIZED
    SIMD_LOOP = discpythonapi.RowType.SIMD_LOOP
    NOT_EXECUTED_LOOP = discpythonapi.RowType.NOT_EXECUTED_LOOP
    FULLY_UNROLLED_LOOP = discpythonapi.RowType.FULLY_UNROLLED_LOOP
    INSIDE_VECTORIZED_LOOP = discpythonapi.RowType.INSIDE_VECTORIZED_LOOP
    THREADED_AUTO_LOOP = discpythonapi.RowType.THREADED_AUTO_LOOP
    THREADED_OPENMP_LOOP = discpythonapi.RowType.THREADED_OPENMP_LOOP
    THREADED_SILK_LOOP = discpythonapi.RowType.THREADED_SILK_LOOP
    INLINED_FUNCTION = discpythonapi.RowType.INLINED_FUNCTION
    LIBRARY_INTERNAL = discpythonapi.RowType.LIBRARY_INTERNAL
    GPU_COMPUTE_TASK = discpythonapi.RowType.GPU_COMPUTE_TASK

class DeviceType(object):
    UNDEFINED = discpythonapi.DeviceType.UNDEFINED
    CPU = discpythonapi.DeviceType.CPU
    GPU = discpythonapi.DeviceType.GPU
    D_GPU = discpythonapi.DeviceType.D_GPU
    END = discpythonapi.DeviceType.END

class DataType(object):
    TOP_DOWN = discpythonapi.DataType.TOP_DOWN
    BOTTOM_UP = discpythonapi.DataType.BOTTOM_UP

class RoofType(object):
    SINGLE_THREAD = discpythonapi.RoofType.SINGLE_THREAD
    MULTI_THREAD = discpythonapi.RoofType.MULTI_THREAD
    SINGLE_NODE = discpythonapi.RoofType.SINGLE_NODE
    COMPUTE = discpythonapi.RoofType.COMPUTE
    MEMORY = discpythonapi.RoofType.MEMORY
    SCALAR = discpythonapi.RoofType.SCALAR

Hotspot = discpythonapi.Hotspot
BasicBlock = discpythonapi.BasicBlock
GpuRow = discpythonapi.GpuRow
PerfProjectionRow = discpythonapi.PerfProjectionRow
RoofItem = discpythonapi.RoofItem

# versioning
doc = """
API and product versions
Returns:
    dictionary of two elements: 'api' and 'product' with versions represented as
    dictionary with 'major', 'minor, 'revision' (and 'build' for product) keys.
"""
get_versions = discpythonapi.get_versions

# deprecated columns
deprecated_callables = set()  # required to keep pointers to callables for deprecated columns, otherwise garbage collector can remove them
def add_deprecated_column(name, replacement, since_major, since_minor, remove_major, remove_minor,
                          string_callable=lambda x: x, type_callable=lambda x: x):
    """
    Add a column to the list of deprecations.

    Args:
        name: column name
        replacement: new column name which should be used instead of deprecated one
        since_major: column is deprecated since this major version of API
        since_minor: column is deprecated since this minor version of API
        remove_major: column will be removed from this major version of API
        remove_minor: column will be removed from this minor version of API
        string_callable: function to process data from `replacement` column when accessing using []
        type_callable: function to process data from `replacement` column when accessing using attribute

    Examples:
        >>> wrap = lambda x: '__' + x + '__'
        >>> add_deprecated_column('my_function', 'function_call_sites_and_loops', 0, 11, 1, 0, string_callable=wrap, type_callable=wrap)
        >>> add_deprecated_column('my_self_time', 'self_time', 0, 11, 1, 0, string_callable=lambda x: str(float(x) * 10), type_callable=lambda x: x * 10)
    """
    global deprecated_callables
    deprecated_callables.add(string_callable)
    deprecated_callables.add(type_callable)
    discpythonapi.add_deprecated_column(name, replacement, since_major, since_minor, remove_major, remove_minor,
                                        string_callable, type_callable)

add_deprecated_column('total_memory_loads', 'all_memory_loads', 0, 11, 0, 15)
add_deprecated_column('total_memory_stores', 'all_memory_stores', 0, 11, 0, 15)
add_deprecated_column('total_cache_misses', 'all_cache_misses', 0, 11, 0, 15)
add_deprecated_column('total_rfo_cache_misses', 'all_rfo_cache_misses', 0, 11, 0, 15)
add_deprecated_column('total_dirty_evictions', 'all_dirty_evictions', 0, 11, 0, 15)

del discpythonapi


# creating properties
doc = 'Equivalent of :attr:`DataModel.bottomup_flat`'
DataModel.bottomup = property(DataModel.get_bottomup_rows, doc=doc)
doc = 'Iterator over :class:`Hotspot` bottom-up rows in flat mode without virtual loops.'
DataModel.bottomup_flat = property(DataModel.get_bottomup_rows, doc=doc)
doc = 'Iterator over :class:`Hotspot` bottom-up rows.'
DataModel.bottomup_full = property(DataModel.get_full_bottomup, doc=doc)
doc = 'Iterator over :class:`Hotspot` top-down rows.'
DataModel.topdown = property(DataModel.get_topdown_rows, doc=doc)
doc = 'Iterator over :class:`GpuRow` gpu rows.'
DataModel.gpu = property(lambda DataModel: DataModel.get_gpu_rows(GpuDataType.BASIC), doc=doc)
doc = 'Iterator over Dependencies :class:`Site` rows.'
DataModel.dependencies = property(DataModel.get_dependencies, doc=doc)
doc = 'Iterator over MAP :class:`Site` rows.'
DataModel.map = property(DataModel.get_map, doc=doc)
doc = 'Iterator over :class:`PerfProjectionRow` rows.'
DataModel.perf_projection = property(DataModel.get_perf_projection_rows, doc=doc)
doc = 'Iterator over :class:`RoofItem` rows.'
DataModel.roofs = property(DataModel.get_roofs, doc=doc)
doc = 'Iterator over :class:`RoofItem` rows.'
DataModel.gpu_roofs = property(DataModel.get_gpu_roofs, doc=doc)
doc = 'Per program metrics, :class:`ProgramMetrics`'
DataModel.metrics = property(DataModel._get_program_metrics, doc=doc)

@deprecated('All memory levels are available at once. This function do nothing.')
def set_memory_level(self, x): pass
DataModel.set_memory_level = set_memory_level

doc = """
Main class to work with the loaded data

Contains the following row iterators:

* :attr:`DataModel.bottomup`
* :attr:`DataModel.topdown`
* :attr:`DataModel.map`
* :attr:`DataModel.dependencies`
"""
DataModel.__doc__ = doc

doc = """
:class:`Project` class is used to load results (:meth:`Project.load` method), to select loops for deeper analysis (**Project.mark_up_\*** methods) and to collect data (:meth:`Project.collect` method, :meth:`Project.collect_ex` method).
"""
Project.__doc__ = doc

doc = 'Iterator over :class:`Site` rows.'
Hotspot.dependencies = property(Hotspot.get_dependencies, doc=doc)

doc = 'Iterator over :class:`Site` rows.'
Hotspot.map = property(Hotspot.get_map, doc=doc)

doc = ':class:`Hotspot` parent row.'
Hotspot.parent = property(Hotspot.get_parent, doc=doc)

doc = 'Iterator over :class:`Hotspot` children rows.'
Hotspot.children = property(Hotspot.get_children, doc=doc)

doc = 'Iterator over :class:`Hotspot` rows from another dataset in pair (bottomup, topdown)corresponding to this row. If row is from the bottomup dataset then sync returns corresponding rows from the topdown and vice versa.'
Hotspot.sync = property(Hotspot.get_sync, doc=doc)

doc = 'Iterator over :class:`Hotspot` basic blocks with additional info.'
Hotspot.basic_blocks = property(lambda slf: slf.get_basic_blocks(True), doc=doc)

doc = 'Iterator over :class:`Hotspot` basic blocks without additional info.'
Hotspot.basic_blocks_lite = property(lambda slf: slf.get_basic_blocks(False), doc=doc)


Hotspot.get_instruction_mix = get_instruction_mix

doc = 'Iterator over :class:`InstructionMixLine` for exclusive (self) statically calculated instruction mix data.'
Hotspot.static_self_mix = property(lambda slf: slf.get_instruction_mix(MixType.STATIC_SELF), doc=doc)

doc = 'Iterator over :class:`InstructionMixLine` for inclusive (aggregated) statically calculated instruction mix data.'
Hotspot.static_total_mix = property(lambda slf: slf.get_instruction_mix(MixType.STATIC_TOTAL), doc=doc)

doc = 'Iterator over :class:`InstructionMixLine` for exclusive (self) dynamically calculated instruction mix data.'
Hotspot.dynamic_self_mix = property(lambda slf: slf.get_instruction_mix(MixType.DYNAMIC_SELF), doc=doc)

doc = 'Iterator over :class:`InstructionMixLine` for inclusive (aggregated) dynamically calculated instruction mix data including top-down connected metrics.'
Hotspot.dynamic_total_mix = property(lambda slf: slf.get_instruction_mix(MixType.DYNAMIC_TOTAL), doc=doc)

doc = 'Iterator over :class:`MemoryObjectsLine` for accessed heap objects data.'
Hotspot.memory_objects = property(Hotspot.get_memory_objects, doc=doc)

doc = 'Iterator over rows (:class:`DatasetRow`) in CPU kernel info dataset.'
Hotspot.cpu_kernel_info = property(lambda slf: slf.get_cpu_kernel_info(), doc=doc)

def _asm_proxy(slf):
    from . import asm
    res = asm.AssemblyIterator(slf.get_assembly())
    del asm
    return res
doc = 'Iterator over :class:`Hotspot` assembly rows.'
Hotspot.assembly = property(_asm_proxy, doc=doc)

doc = 'Iterator over :class:`GpuRow` child rows.'
GpuRow.children = property(GpuRow.get_children, doc=doc)

doc = ':class:`GpuRow` parent row.'
GpuRow.parent = property(GpuRow.get_parent, doc=doc)

doc = ':class:`GpuRow` row memory objects.'
GpuRow.memory_objects = property(GpuRow.get_memory_objects, doc=doc)

doc = 'Iterator over :class:`PerfProjectionRow` child rows.'
PerfProjectionRow.children = property(PerfProjectionRow.get_children, doc=doc)

doc = ':class:`PerfProjectionRow` parent row.'
PerfProjectionRow.parent = property(PerfProjectionRow.get_parent, doc=doc)

doc = 'Iterator over :class:`DatasetRow` instruction mix dataset for performance projection.'
PerfProjectionRow.instruction_mix = property(lambda PerfProjectionRow: PerfProjectionRow._get_additional_data(AdditionalDataType.INSTRUCTION_MIX), doc=doc)

doc = 'Iterator over :class:`DatasetRow` memory traffic dataset for performance projection.'
PerfProjectionRow.memory_traffic = property(lambda PerfProjectionRow: PerfProjectionRow._get_additional_data(AdditionalDataType.MEMORY_TRAFFIC), doc=doc)

doc = 'Iterator over :class:`DatasetRow` memory objects dataset for performance projection.'
PerfProjectionRow.memory_objects = property(lambda PerfProjectionRow: PerfProjectionRow._get_additional_data(AdditionalDataType.MEMORY_OBJECTS), doc=doc)

doc = ':class:`Hotspot` row from top-down dataset corresponding to this row.'
PerfProjectionRow.top_down = property(lambda PerfProjectionRow:
    next(PerfProjectionRow._get_linked_hotspots_rows(DataType.TOP_DOWN), None), doc=doc)

doc = ':class:`Hotspot` row from bottom-up dataset corresponding to this row.'
PerfProjectionRow.bottom_up = property(lambda PerfProjectionRow:
    next(PerfProjectionRow._get_linked_hotspots_rows(DataType.BOTTOM_UP), None), doc=doc)

doc = 'Iterator over rows (:class:`DatasetRow`) in CPU kernel info dataset for performance projection.'
PerfProjectionRow.cpu_kernel_info = property(lambda PerfProjectionRow: PerfProjectionRow._get_additional_data(AdditionalDataType.CPU_KERNEL_INFO), doc=doc)

doc = 'Iterator over :class:`Problem` rows.'
Site.problems = property(Site.get_problems, doc=doc)

doc = 'Iterator over :class:`Observation` rows.'
Problem.observations = property(Problem.get_observations, doc=doc)

doc = 'Iterator over :class:`BasicBlock` assembly rows.'
BasicBlock.assembly = property(_asm_proxy, doc=doc)

def _get_map_db_path(data_model):
    import glob
    import os
    result_path = data_model._get_map_result_path()
    dbfile_pattern = os.path.join(result_path, 'data.0', '*.db3')
    files = glob.glob(dbfile_pattern)
    if files:
        return files[0]
    print('Can\'t find map db3 file')
    return ''


@deprecated('Use corresponding MAP site columns instead.')
def _get_cachesim_info(self, site_id):
    """
    Get cache simulator info for one site of collected MAP result.

    Args:
        site_id: MAP site id
    Returns:
        :class:`CacheSimulatorInfo` instance
    """
    from . import cachesim
    dbfile = _get_map_db_path(self)
    return cachesim.CacheSimulatorInfo(dbfile, site_id)

DataModel.get_cachesim_info = _get_cachesim_info

from .getcfg import _get_cfg_info
DataModel.get_cfg_info = _get_cfg_info

from .getcfg import _get_runtool_cfg
DataModel.get_runtool_cfg = _get_runtool_cfg

RoofItem.has_type = lambda self, type: (self.types & type) != 0

from . import mark_up
mark_up.row_type = Hotspot
Project.mark_up_select              = mark_up.select
Project.mark_up_append              = mark_up.append
Project.mark_up_remove              = mark_up.remove
Project.mark_up_select_for_target   = mark_up.select_for_target
doc = """
Number of selected loops.

Examples:
    >>> project.select(rows[:3])
    >>> project.mark_up_len
    3
"""
Project.mark_up_len = property(Project._get_selected_loops_num, doc=doc)
del mark_up


from . import collect
collect.advixe_cl_path = advixe_cl_path
Project.collect = collect.collect
Project.collect_ex = collect.collect_ex
del advixe_cl_path
del collect

from . import html_report

del doc
