# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import logging
import os
from pathlib import Path
from typing import Union, Dict, Any, List

from mpp import MPP_ROOT_DIR
from mpp.console_output import ConsoleOutput
from mpp.core.types import ConfigurationPaths, VerboseLevel


class ConfigurationPathGenerator:
    CONFIGURATION_PATH = [ConfigurationPaths.METRIC_PATH, ConfigurationPaths.CHART_PATH]
    DIRECTORY_KEY = 'directory'

    def __init__(self,
                 configuration_file_paths: Dict[Any, Dict[str, Path]],
                 config_file_path_args: List[Union[Dict[str, Path], None]],
                 valid_core_types: List[str],
                 verbose: VerboseLevel = VerboseLevel.INFO):
        self.configuration_file_paths = configuration_file_paths
        self.config_file_path_args = config_file_path_args
        self.__valid_core_types = valid_core_types
        self.__num_files_not_found_per_core_type = {core_type: 0 for core_type in self.__valid_core_types}
        self.__verbose = verbose
        self.__validate_preconditions()
        self.__filter_invalid_core_types()
        self.__reformat_config_file_path_args()
        self.__merge_configuration_args()
        self.__handle_null_configuration_file_paths()

    def generate(self):
        updated_configuration_file_paths = {}
        for core_type, core_type_path in self.configuration_file_paths.items():
            for config_arg_name in self.config_file_path_args[core_type]:
                config_path = core_type_path[config_arg_name]
                updated_configuration_file_paths = (
                    self.__update_configuration_file_path(config_arg_name, core_type, config_path,
                                                          updated_configuration_file_paths))
        updated_configuration_file_paths = self.__delete_core_types_with_missing_metric_files(
            updated_configuration_file_paths)
        self.__validate_results()
        return updated_configuration_file_paths

    def __validate_preconditions(self):
        if self.__edp_configuration_section_not_found():
            if self.__any_config_arg_is_passed():
                raise InvalidConfigurationArgumentError(
                    'EDP Configuration section does not exist in your -i input file, -m must have a valid '
                    'metric file path and -f (if provided) must have a valid chart format file path.')

    def __validate_results(self):
        if self.__all_core_types_are_missing_a_file():
            raise ConfigurationFileNotFoundError(message='Input file path is not found, all core types are missing at '
                                                         'least one configuration file (-m and/or -f)')

    def __all_core_types_are_missing_a_file(self):
        return all(
            [num_files_not_found > 0 for num_files_not_found in self.__num_files_not_found_per_core_type.values()])

    def __filter_invalid_core_types(self):
        for core_type in self.configuration_file_paths.copy():
            if core_type not in self.__valid_core_types:
                self.configuration_file_paths.pop(core_type)

    def __reformat_config_file_path_args(self):
        for file_path_arg in self.config_file_path_args:
            if not file_path_arg:
                continue
            for core_type in self.__valid_core_types:
                if self.DIRECTORY_KEY in file_path_arg:
                    file_path_arg[core_type] = file_path_arg.get(core_type, file_path_arg[self.DIRECTORY_KEY])
                else:
                    file_path_arg[core_type] = file_path_arg.get(core_type, {})
            if self.DIRECTORY_KEY in file_path_arg:
                file_path_arg.pop(self.DIRECTORY_KEY)

    def __edp_configuration_section_not_found(self):
        return not self.configuration_file_paths

    def __any_config_arg_is_passed(self):
        return {} in self.config_file_path_args

    def __merge_configuration_args(self):
        config_file_path_args = {}
        config_file_path_args.update({core_type: {} for core_type in self.__valid_core_types})
        for core_type in self.__valid_core_types:
            for idx, arg in enumerate(self.config_file_path_args):
                if arg is not None:
                    if arg == {} or arg[core_type] == {}:
                        config_file_path_args[core_type].update({self.CONFIGURATION_PATH[idx]: None})
                        continue
                    if not isinstance(arg[core_type], Path):
                        arg[core_type] = Path(arg[core_type])
                    config_file_path_args[core_type].update({self.CONFIGURATION_PATH[idx]: arg[core_type]})
        self.config_file_path_args = config_file_path_args

    def __handle_null_configuration_file_paths(self):
        if not self.configuration_file_paths:
            self.configuration_file_paths = {core_type: {
                ConfigurationPaths.METRIC_PATH: None,
                ConfigurationPaths.CHART_PATH: None
            } for core_type in self.__valid_core_types}

    def __update_configuration_file_path(self, config_arg_name, core_type, config_path,
                                         updated_configuration_file_paths):
        config_file_path_arg = self.config_file_path_args[core_type][config_arg_name]
        if core_type not in updated_configuration_file_paths:
            updated_configuration_file_paths[core_type] = {}
        config_file_finder_factory = _ConfigurationFileFinderFactory(config_file_path_arg)
        config_file_finder = config_file_finder_factory.create(config_path)
        try:
            updated_configuration_file_paths[core_type][config_arg_name] = config_file_finder.find()
            if ConsoleOutput.is_regular_verbosity(self.__verbose):
                print(f'\n{config_file_finder_factory.config_arg_verb} `{config_arg_name}` for {core_type} at:'
                      f' {updated_configuration_file_paths[core_type][config_arg_name]}')
        except ConfigurationFileNotFoundError as e:
            self._handle_core_type_path_missing(config_arg_name, core_type, e)
            self.__set_config_arg_name_to_none(config_arg_name, core_type, updated_configuration_file_paths)
        return updated_configuration_file_paths

    def _handle_core_type_path_missing(self, config_arg_name, core_type, e):
        split_delimiter = '\n(Search Path'
        error_string = str(e).split(split_delimiter)
        logging.warning(error_string[0].replace('Input file path', f'`{config_arg_name.capitalize()}`') +
                        f' for `{core_type}` core type ' + split_delimiter + error_string[1])
        self.__num_files_not_found_per_core_type[core_type] += 1

    @staticmethod
    def __set_config_arg_name_to_none(config_arg_name, core_type, updated_configuration_file_paths):
        if core_type in updated_configuration_file_paths:
            updated_configuration_file_paths[core_type][config_arg_name] = None

    @staticmethod
    def __delete_core_types_with_missing_metric_files(updated_configuration_file_paths):
        for core_type, core_type_path in updated_configuration_file_paths.copy().items():
            if ConfigurationPaths.METRIC_PATH not in core_type_path.keys():
                del updated_configuration_file_paths[core_type]
        return updated_configuration_file_paths


class _ConfigurationFileFinderFactory:

    def __init__(self, config_file_path_arg):
        self.config_arg_verb = 'Found'
        self.config_file_path_arg = config_file_path_arg

    def create(self, config_path):
        if self.__config_path_arg_is_not_provided():
            return _DefaultConfigurationFileFinder(
                config_path)
        elif self.__config_path_arg_is_a_directory():
            return _DirectoryConfigurationFileFinder(
                config_path, self.config_file_path_arg)
        elif self.__config_path_arg_is_a_file():
            self.config_arg_verb = 'Using'
            return _FileConfigurationFileFinder(
                config_path, self.config_file_path_arg)
        elif self.__config_path_does_not_exist():
            raise ConfigurationFileNotFoundError(message=f'Configuration path does not exist: '
                                                         f'{self.config_file_path_arg}')

    def __config_path_arg_is_not_provided(self):
        return self.config_file_path_arg is None

    def __config_path_arg_is_a_directory(self):
        return self.config_file_path_arg.is_dir()

    def __config_path_arg_is_a_file(self):
        return self.config_file_path_arg.is_file()

    def __config_path_does_not_exist(self):
        return not os.path.exists(self.config_file_path_arg)


class _DefaultConfigurationFileFinder:

    def __init__(self, config_file_path, config_file_path_arg=None):
        self.config_file_path = config_file_path
        self.config_file_path_arg = config_file_path_arg
        self.updated_configuration_file_path = None
        self.search_path = None
        self.search_string = ''

    def find(self):
        self._set_search_arguments()
        self.updated_configuration_file_path = self._get_updated_configuration_file_path()
        config_file_validator = _ConfigurationPathValidator(self.updated_configuration_file_path)
        config_file_validator.validate(self.search_path, self.search_string)
        return self.updated_configuration_file_path

    def _set_search_arguments(self):
        self.search_path = MPP_ROOT_DIR
        self.search_string = 'current or parent directory'

    def _get_updated_configuration_file_path(self):
        parent_directory = self.search_path.parent
        configuration_file_path = self._search_directory_for_file(self.search_path)
        if configuration_file_path:
            return configuration_file_path
        configuration_file_path = self._search_directory_for_file(parent_directory)
        return configuration_file_path

    def _search_directory_for_file(self, directory: Path):
        for file in directory.glob(str(self.config_file_path)):
            return file


class _DirectoryConfigurationFileFinder(_DefaultConfigurationFileFinder):

    def _set_search_arguments(self):
        self.search_path = self.config_file_path_arg
        self.search_string = 'provided directory'

    def _get_updated_configuration_file_path(self):
        configuration_file_path = self._search_directory_for_file(self.search_path)
        return configuration_file_path


class _FileConfigurationFileFinder(_DefaultConfigurationFileFinder):

    def _set_search_arguments(self):
        self.search_path = self.config_file_path_arg
        self.search_string = 'provided file'

    def _get_updated_configuration_file_path(self):
        return self.config_file_path_arg


class _ConfigurationPathValidator:

    def __init__(self, configuration_file_path):
        self.configuration_file_path = configuration_file_path

    def validate(self, search_path=None, search_string=''):
        if self.__path_not_found():
            raise ConfigurationFileNotFoundError(search_string, search_path)
        if self.__path_does_not_exist():
            raise ConfigurationFileNotFoundError(search_string, search_path)

    def __path_does_not_exist(self):
        return not os.path.exists(self.configuration_file_path)

    def __path_not_found(self):
        return not self.configuration_file_path


class ConfigurationFileNotFoundError(FileNotFoundError):
    def __init__(self, search_string='', search_path='', message=''):
        if not message:
            message = (f'Input file path is not found in {search_string}\n'
                       f'(Search Path: {search_path})')
        super().__init__(message)


class InvalidConfigurationArgumentError(ValueError):
    def __init__(self, message='Invalid configuration argument provided'):
        super().__init__(message)
