# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

from abc import ABC
from dataclasses import dataclass
from pathlib import Path

from mpp import ViewAttributes, ViewType
from tools.common.validators import FileValidator


@dataclass
class ExcelFormat():
    font_name: str = 'Courier New'
    font_size: int = 10
    number_format: str = '#,##0.000'


class ExcelSheet(ABC):
    MAX_CSV_SIZE = 100 * 1024 * 1024 * 1024

    def __init__(self, csv_dir: str, view_attributes: ViewAttributes, tab_color: str,
                 fmt: ExcelFormat = ExcelFormat(), include_charts=False):

        self.__file_path = str(csv_dir / Path(f'{view_attributes.view_name}.csv'))
        self._validate_file()
        self.__view_attributes = view_attributes
        self.__tab_color: str = tab_color
        self.__format = fmt
        self.__include_charts = include_charts
        self.__sheet_name = self._get_sheet_name()

    def _validate_file(self):
        file_validator = FileValidator(file_must_exist=True, max_file_size=self.MAX_CSV_SIZE)
        file_validator(self.__file_path)

    @property
    def file_path(self) -> str:
        return self.__file_path

    @property
    def tab_color(self) -> str:
        return self.__tab_color

    @property
    def fmt(self) -> ExcelFormat:
        return self.__format

    @property
    def include_charts(self) -> bool:
        return self.__include_charts

    @property
    def name(self) -> str:
        return self.__sheet_name

    @property
    def type(self) -> ViewType:
        return self.__view_attributes.view_type

    @staticmethod
    def _get_view_type_str_for_sheet_name() -> str:
        return ''

    def _get_sheet_name(self):
        device_prefix = self.__get_device_prefix()
        aggregation_level = self.__view_attributes.aggregation_level.name.lower()
        view_type = self._get_view_type_str_for_sheet_name()
        sheet_name = f'{device_prefix}{view_type}{aggregation_level} view'
        sheet_name.strip()
        return sheet_name

    def __get_device_prefix(self) -> str:
        device = self.__view_attributes.device.label
        return '' if device == '' else device + ' '


class SummarySheet(ExcelSheet):
    COLOR = '#006300'
    FORMAT = ExcelFormat('Courier New', 10, '#,##0.000')

    def __init__(self, file_path: str, view_attributes: ViewAttributes):
        super().__init__(file_path, view_attributes, self.COLOR)


class TpsSummarySheet(ExcelSheet):
    COLOR = '#000000'
    FORMAT = ExcelFormat('Courier New', 10, '#,##0.000')
    PER_TXN_POSTFIX = ' (per-txn)'

    def __init__(self, file_path: str, view_attributes: ViewAttributes):
        super().__init__(file_path, view_attributes, self.COLOR)

    def _get_sheet_name(self) -> str:
        return super()._get_sheet_name() + self.PER_TXN_POSTFIX


class DetailsSheet(ExcelSheet):
    COLOR = '#630000'
    FORMAT = ExcelFormat('Calibri', 8, '#,##0.000')
    DETAILS_STR_FOR_SHEET_NAME = 'details '

    def __init__(self, file_path: str, view_attributes: ViewAttributes, include_charts=False):
        super().__init__(file_path, view_attributes, self.COLOR, self.FORMAT, include_charts)

    @staticmethod
    def _get_view_type_str_for_sheet_name() -> str:
        return DetailsSheet.DETAILS_STR_FOR_SHEET_NAME


class ExcelSheetFactory:

    @staticmethod
    def create(file_path: str, view_attributes: ViewAttributes, include_details, include_charts) -> ExcelSheet:
        view_type = view_attributes.view_type
        if view_type == ViewType.SUMMARY:
            return SummarySheet(file_path, view_attributes)
        elif view_type == ViewType.TPS_SUMMARY:
            return TpsSummarySheet(file_path, view_attributes)
        elif view_type == ViewType.DETAILS and include_details:
            return DetailsSheet(file_path, view_attributes, include_charts)
        else:
            view_name = view_attributes.view_name
            raise ValueError(f'Unsupported view type, {view_type}, for {view_name}')
