#!/bin/sh
#
# Copyright (C) 2019 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

#
# Description:
#
#     Create a copy of VTune(TM) Amplifier Perf tool with performance monitoring
#     privileges and limit access to the new executable for other users in the
#     system who are not in the specified group.
#
#     Following capabilities will be set for the new executable:
#         'cap_sys_admin,cap_sys_ptrace,cap_syslog=ep,cap_perfmon';
#          Either one of 'cap_sys_admin' or 'cap_perfmon' will be set;
#          Additionally, 'cap_syslog' will be set if the kernel supports it;
#
#          Note: 'cap_sys_ptrace' could be ommited on kernels which doesn't
#                require it for context switches;
#
#     Prerequisites:
#         - extended attributes are supported by the file system;
#         - 'setcap' command is available;
#

SCRIPT_DIR=$(cd -P "$(dirname "$0")" && pwd)
PERF_EXE="amplxe-perf"
PERF_EXE_PRIV="amplxe-perf-priv"
GROUP_NAME="vtune"
COMMAND_NAME="setup"
VARIANT="cap_perfmon"

print_msg()
{
    MSG="$*"
    echo "$MSG"
}

print_nnl()
{
    MSG="$*"
    echo -n "$MSG"
}

print_err()
{
    MSG="$*"
    if [ -w /dev/stderr ] ; then
        echo "$MSG" >> /dev/stderr
    else
        echo "$MSG"
    fi
}

print_usage_and_exit()
{
    err=${1:-0}
    print_msg "This script sets Linux capabilities 'CAP_PERFMON, CAP_SYS_PTRACE, CAP_SYSLOG' by default"
    print_msg "or 'cap_sys_admin, CAP_SYS_PTRACE, CAP_SYSLOG' to allow perf collection"
    print_msg ""
    print_msg "Usage: $0 [ options ]"
    print_msg ""
    print_msg " where \"options\" are:"
    print_msg ""
    print_msg "    -h | --help"
    print_msg "      prints out usage"
    print_msg ""
    print_msg "    -v | --variant <capability>"
    print_msg "      specify the capability: 'cap_perfmon' or 'cap_sys_admin'; if this option"
    print_msg "      is not specified, it defaults to 'cap_perfmon'"
    print_msg ""
    print_msg "    -g | --group <group>"
    print_msg "      restricts access to the privileged executable to users in the specified"
    print_msg "      group; if this option is not provided, the group '$GROUP_NAME' will be used"
    print_msg ""
    print_msg "    -c | --command <command>"
    print_msg "      use this option to specify action: 'setup' or 'remove'; if this option"
    print_msg "      is not specified, it defaults to 'setup'"
    print_msg ""
    exit $err
}

create_binary_and_set_capabilities()
{
    BIN_DIR="$(dirname "${SCRIPT_DIR}")/$1"
    PRIVILIGED_BINARY="${BIN_DIR}/${PERF_EXE_PRIV}"

    if ! [ -x "$(command -v setcap)" ] ; then
        print_err "ERROR: unable to find command \"setcap\"."
        print_err ""
        print_err "Please install:"
        print_err ""
        print_err "    \"libcap\" package for Red Hat Enterprise Linux/CentOS/Fedora distributions"
        print_err "    \"libcap2-bin\" package for Ubuntu/Debian distributions"
        print_err "    \"libcap-progs\" package for SLES/openSUSE distributions"
        print_err ""
        exit 255
    fi

    if [ ! -f "${BIN_DIR}/${PERF_EXE}" ] ; then
        echo "ERROR: the binary file \"${BIN_DIR}/${PERF_EXE}\" does not exist."
        exit 102
    fi

    cp "${BIN_DIR}/${PERF_EXE}" "${PRIVILIGED_BINARY}"
    if [ $? -ne 0 ]; then
        print_err "ERROR: failed to create \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
        exit 103
    fi

    chgrp ${GROUP_NAME} "${PRIVILIGED_BINARY}"
    if [ $? -ne 0 ]; then
        print_err "ERROR: failed to set group for \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
        rm "${PRIVILIGED_BINARY}"
        exit 104
    fi

    chmod o-rwx,g+x "${PRIVILIGED_BINARY}"
    if [ $? -ne 0 ]; then
        print_err "ERROR: failed to chmod \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
        rm "${PRIVILIGED_BINARY}"
        exit 105
    fi

    if [ "${VARIANT}" = "cap_sys_admin" ]; then
        setcap "cap_sys_admin,cap_sys_ptrace,cap_syslog=ep" "${PRIVILIGED_BINARY}" >/dev/null 2>&1
        if [ $? -ne 0 ]; then
            setcap "cap_sys_admin,cap_sys_ptrace=ep" "${PRIVILIGED_BINARY}"
            if [ $? -ne 0 ]; then
                print_err "ERROR: failed to assign the required capabilities to \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
                rm "${PRIVILIGED_BINARY}"
                exit 106
            else
                print_msg "cap_sys_admin, cap_sys_ptrace set successfully"
            fi
        else
            print_msg "cap_sys_admin, cap_sys_ptrace, cap_syslog set successfully"
        fi
    fi

    if [ "${VARIANT}" = "cap_perfmon" ]; then
        setcap "cap_perfmon,cap_sys_ptrace,cap_syslog=ep" "${PRIVILIGED_BINARY}" >/dev/null 2>&1
        if [ $? -ne 0 ]; then
            setcap "cap_perfmon,cap_sys_ptrace=ep" "${PRIVILIGED_BINARY}" >/dev/null 2>&1
            if [ $? -ne 0 ]; then
                print_err "ERROR: failed to assign the required capabilities to \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
                print_err ""
                print_err "CAP_PERFMON is probably not supported in this version of Linux kernel or Libcap utilities."
                print_err "Make sure your Linux kernel version is 5.8 or higher, and Libcap utilities version is 2.36 or higher."
                print_err ""
                print_err "Try setting CAP_SYS_ADMIN by using the option: -v cap_sys_admin."
                print_err ""
                rm "${PRIVILIGED_BINARY}"
                exit 106
            else
                print_msg "cap_perfmon, cap_sys_ptrace set successfully"
            fi
        else
            print_msg "cap_perfmon, cap_sys_ptrace, cap_syslog set successfully"
        fi
    fi
}

remove_binary()
{
    BIN_DIR="$(dirname "${SCRIPT_DIR}")/$1"
    PRIVILIGED_BINARY="${BIN_DIR}/${PERF_EXE_PRIV}"

    if [ -f "${PRIVILIGED_BINARY}" ] ; then
        echo "Removing \"${BIN_DIR}/${PERF_EXE_PRIV}\""
        rm "${PRIVILIGED_BINARY}"
        if [ $? -ne 0 ]; then
            print_err "ERROR: failed to remove \"${BIN_DIR}/${PERF_EXE_PRIV}\"."
            exit 107
        fi
    fi
}

ARGS="$@"
if [ $# -eq 1 ] ; then
    case "$1" in
        -h | --help)
        print_usage_and_exit 0
        ;;
    esac
fi

while [ $# -gt 0 ] ; do
    case "$1" in
        -v | --variant)
            case "$2" in
                cap_sys_admin | cap_perfmon)
                VARIANT="$2"
                shift
                ;;
            *)
                print_err ""
                print_err "ERROR: unrecognized variant $2."
                print_usage_and_exit 100
                ;;
            esac

            shift
            ;;

        -g | --group)
            GROUP_NAME="$2"
            shift
            shift
            ;;

        -c | --command)
            case "$2" in
                setup | remove)
                 COMMAND_NAME="$2"
                 shift
                 ;;
            *)
                print_err ""
                print_err "ERROR: unrecognized command $2."
                print_usage_and_exit 100
                ;;
            esac

            shift
            ;;
        *)
            print_err ""
            print_err "ERROR: unrecognized option $1."
            print_usage_and_exit 101
            ;;
    esac
done

if [ $COMMAND_NAME = "setup" ] ; then
    if [ $(id -u) -ne 0 ] ; then
        print_msg "NOTE: super-user or \"root\" privileges are required."
        print_nnl "Please enter \"root\" "
        exec su -c "/bin/sh $0 $ARGS"
        print_msg ""
        exit 0
    fi

    create_binary_and_set_capabilities "bin64"
fi

if [ $COMMAND_NAME = "remove" ] ; then
    remove_binary "bin64"
fi

print_msg "Done"

exit 0
