/*******************************************************************************
* Copyright (C) 2003 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
!  Content:
!    vsRngGaussianMV  Example Program Text
!******************************************************************************/

#include <stdio.h>
#include <math.h>

#include "mkl.h"
#include "errcheck.inc"
#include "statcheck.inc"

#define BRNG    VSL_BRNG_MCG31 // VSL basic generator to be used
#define SEED    7777777        // Initial value for stream initialization
#define NDIM    3              // Number of dimensions
#define N       10000          // Number of NDIM-dimensional vectors to generate
#define NN      5              // Number of NDIM-dimensional vectors to print

/*****************************************************
 Variance-covariance matrix for test
 (should be symmetric,positive-definite)
*****************************************************/
/* This is packed storage for spptrf subroutine */
static float C[NDIM * (NDIM + 1) / 2] = { 16.0f, 8.0f, 4.0f, 13.0f, 17.0f, 62.0f };

/*****************************************************
 Variance-covariance matrix for test
 (should be symmetric,positive-definite)
*****************************************************/
static float a[NDIM]={ 3.0f, 5.0f, 2.0f };
static float r[N][NDIM];

int main(void)
{
    VSLStreamStatePtr stream;

    float  CC[NDIM][NDIM]; // Copy of C
    int i, j, k, errcode = VSL_ERROR_OK, status = 0;

    /* Following variables are used in Cholesky factorization subroutine */
    char    uplo;
    MKL_INT n;
    float* T;
    MKL_INT info;

    /* Sample characteristics */
    float dbMean[NDIM], dbVar[NDIM];
    float dbCovXY, dbCovXZ, dbCovYZ;

    float dbS[NDIM], dbS2[NDIM];

    float S[NDIM], D2[NDIM], Q[NDIM];
    float DeltaM[NDIM], DeltaD[NDIM];

    /*****************************************************
     Printing variance-covariance matrix and mean vector
     Also making the copy CC from C
    *****************************************************/
    k = 0;
    for( i = 0; i < NDIM; i++ )
    {
        for( j = i; j < NDIM; j++ )
        {
            CC[i][j] = C[k++];
        }
        for( j = 0; j < i; j++ )
        {
            CC[i][j] = CC[j][i];
        }
    }

    printf("Variance-covariance matrix C:\n");
    for( i = 0; i < NDIM; i++)
    {
        for( j = 0; j < NDIM; j++ )
        {
            printf("%4.1f ",CC[i][j]);
        }
        printf("\n");
    }
    printf("\n");
    printf("Mean vector a:\n");
    for( i = 0; i < NDIM; i++ )
    {
        printf("%4.1f ",a[i]);
    }
    printf("\n\n");

    /*****************************************************
     Cholesky factorization
    *****************************************************/
    /* Packed maxtrix storage (SPPTRF subroutine) */
    uplo = 'L';
    n = NDIM;
    T = (float*)C;

    spptrf( &uplo, &n, T, &info );

    printf("VSL_MATRIX_STORAGE_PACKED\n");
    printf("-------------------------\n\n");

    /*****************************************************
     Stream initialization
    *****************************************************/
    errcode = vslNewStream( &stream, BRNG, SEED );
    CheckVslError( errcode );

    /*****************************************************
     Generating random numbers
     from multivariate normal distribution
    *****************************************************/
    errcode = vsRngGaussianMV( VSL_RNG_METHOD_GAUSSIANMV_BOXMULLER2, stream, N, (float *)r, NDIM, VSL_MATRIX_STORAGE_PACKED, a, T );
    CheckVslError( errcode );

    /*****************************************************
     Printing random numbers
    *****************************************************/
    printf("Results (first %d of %d):\n", NN, N);
    printf("---------------------------\n");
    for( i = 0; i < NN; i++ )
    {
        printf("r[%d]=(",i);
        for( j = 0; j < NDIM; j++ )
        {
            printf("%5.2f ",r[i][j]);
        }
        printf(")\n");
    }
    printf("\n");

    /*****************************************************
     Testing sample characteristics
    *****************************************************/
    sCalculateGaussianMVSampleCharacteristics(N, NDIM, (float *)r,
            dbS, dbS2, dbMean, dbVar, &dbCovXY, &dbCovXZ, &dbCovYZ);

    /* Printing results */
    printf("Sample characteristics:\n");
    printf("-----------------------\n");
    printf("       Sample           Theory\n");
    printf("Mean :(%4.1f,%4.1f,%4.1f) (%4.1f,%4.1f,%4.1f)\n",dbMean[0],dbMean[1],dbMean[2],a[0],a[1],a[2]);
    printf("Var. :(%4.1f,%4.1f,%4.1f) (%4.1f,%4.1f,%4.1f)\n",dbVar[0],dbVar[1],dbVar[2],CC[0][0],CC[1][1],CC[2][2]);
    printf("CovXY: %4.1f               %4.1f\n",dbCovXY,CC[0][1]);
    printf("CovXZ: %4.1f               %4.1f\n",dbCovXZ,CC[0][2]);
    printf("CovYZ: %4.1f               %4.1f\n\n\n",dbCovYZ,CC[1][2]);

    /* Checking results */
    status = sGaussianMVCheckResults(NDIM, N, a, (float *)CC, dbMean, dbVar, S, D2, Q, DeltaM, DeltaD);

    if( status )
    {
        printf ("Error: sample moments\n");
        printf ("disagree with theory\n");
        printf ("     DeltaM: %7.3f %7.3f %7.3f \n", DeltaM[0], DeltaM[1], DeltaM[2]);
        printf ("     DeltaD: %7.3f %7.3f %7.3f \n", DeltaD[0], DeltaD[1], DeltaD[2]);
        printf ("     ( at least one of the Deltas > 3.0)\n");
    }
    else
    {
        printf ("Sample moments\n");
        printf ("agree with theory\n");
        printf ("     DeltaM: %7.3f %7.3f %7.3f \n", DeltaM[0], DeltaM[1], DeltaM[2]);
        printf ("     DeltaD: %7.3f %7.3f %7.3f \n", DeltaD[0], DeltaD[1], DeltaD[2]);
        printf ("     ( All Deltas < 3.0)\n");
    }

    /*****************************************************
     Stream finalization
    *****************************************************/
    errcode = vslDeleteStream( &stream );
    CheckVslError( errcode );

    return status;
}
