/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.www;

import java.io.IOException;
import java.io.PrintWriter;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.hop.core.Const;
import org.apache.hop.core.annotations.HopServerServlet;
import org.apache.hop.core.logging.HopLogStore;
import org.apache.hop.core.util.Utils;
import org.apache.hop.core.xml.XmlHandler;
import org.apache.hop.i18n.BaseMessages;
import org.apache.hop.workflow.WorkflowMeta;
import org.apache.hop.workflow.engine.IWorkflowEngine;
import org.owasp.encoder.Encode;

@HopServerServlet(id = "removeWorkflow", name = "Remove a workflow from the server")
public class RemoveWorkflowServlet extends BaseHttpServlet implements IHopServerPlugin {

  private static final Class<?> PKG = RemoveWorkflowServlet.class;
  private static final long serialVersionUID = -2051906998698124039L;

  public static final String CONTEXT_PATH = "/hop/removeWorkflow";

  public RemoveWorkflowServlet() {}

  public RemoveWorkflowServlet(WorkflowMap workflowMap) {
    super(workflowMap);
  }

  @Override
  public void doGet(HttpServletRequest request, HttpServletResponse response)
      throws ServletException, IOException {
    if (isJettyMode() && !request.getContextPath().startsWith(CONTEXT_PATH)) {
      return;
    }

    if (log.isDebug()) {
      logDebug(BaseMessages.getString(PKG, "RemoveWorkflowServlet.Log.RemoveWorkflowRequested"));
    }

    String workflowName = request.getParameter("name");
    String id = request.getParameter("id");
    boolean useXML = "Y".equalsIgnoreCase(request.getParameter("xml"));

    response.setStatus(HttpServletResponse.SC_OK);

    if (useXML) {
      response.setContentType("text/xml");
      response.setCharacterEncoding(Const.XML_ENCODING);
    } else {
      response.setContentType("text/html;charset=UTF-8");
    }

    PrintWriter out = response.getWriter();

    // ID is optional...
    //
    IWorkflowEngine<WorkflowMeta> workflow;
    HopServerObjectEntry entry;
    if (Utils.isEmpty(id)) {
      // get the first pipeline that matches...
      //
      entry = getWorkflowMap().getFirstHopServerObjectEntry(workflowName);
      if (entry == null) {
        workflow = null;
      } else {
        id = entry.getId();
        workflow = getWorkflowMap().getWorkflow(entry);
      }
    } else {
      // Take the ID into account!
      //
      entry = new HopServerObjectEntry(workflowName, id);
      workflow = getWorkflowMap().getWorkflow(entry);
    }

    if (workflow != null) {
      HopLogStore.discardLines(workflow.getLogChannelId(), true);
      getWorkflowMap().removeWorkflow(entry);

      if (useXML) {
        response.setContentType("text/xml");
        response.setCharacterEncoding(Const.XML_ENCODING);
        out.print(XmlHandler.getXmlHeader(Const.XML_ENCODING));
        out.print(WebResult.OK.getXml());
      } else {
        response.setContentType("text/html;charset=UTF-8");

        out.println("<HTML>");
        out.println("<HEAD>");
        out.println(
            "<TITLE>"
                + BaseMessages.getString(PKG, "RemoveWorkflowServlet.WorkflowRemoved")
                + "</TITLE>");
        out.println("<META http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">");
        out.println(
            "<link rel=\"icon\" type=\"image/svg+xml\" href=\"/static/images/favicon.svg\">");
        out.println("</HEAD>");
        out.println("<BODY>");
        out.println(
            "<H3>"
                + Encode.forHtml(
                    BaseMessages.getString(
                        PKG, "RemoveWorkflowServlet.TheWorkflowWasRemoved", workflowName, id))
                + "</H3>");
        out.print(
            "<a href=\""
                + convertContextPath(GetStatusServlet.CONTEXT_PATH)
                + "\">"
                + BaseMessages.getString(PKG, "PipelineStatusServlet.BackToStatusPage")
                + "</a><br>");
        out.println("<p>");
        out.println("</BODY>");
        out.println("</HTML>");
      }
    } else {
      if (useXML) {
        out.println(
            new WebResult(
                WebResult.STRING_ERROR,
                BaseMessages.getString(
                    PKG, "RemoveWorkflowServlet.Log.CoundNotFindSpecWorkflow", workflowName)));
      } else {
        out.println(
            "<H1>"
                + Encode.forHtml(
                    BaseMessages.getString(
                        PKG,
                        "RemoveWorkflowServlet.WorkflowRemoved.Log.CoundNotFindWorkflow",
                        workflowName,
                        id))
                + "</H1>");
        out.println(
            "<a href=\""
                + convertContextPath(GetStatusServlet.CONTEXT_PATH)
                + "\">"
                + BaseMessages.getString(PKG, "PipelineStatusServlet.BackToStatusPage")
                + "</a><p>");
        response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
      }
    }
  }

  public String toString() {
    return "Remove workflow servlet";
  }

  @Override
  public String getService() {
    return CONTEXT_PATH + " (" + toString() + ")";
  }

  @Override
  public String getContextPath() {
    return CONTEXT_PATH;
  }
}
